/*
 * @(#)EntitySet.java  1.0  2006-03-20
 *
 * Copyright (c) 2006 Lucerne University of Applied Sciences and Arts (HSLU)
 * Zentralstrasse 18, Postfach 2858, CH-6002 Lucerne, Switzerland
 * All rights reserved.
 *
 * The copyright of this software is owned by the Lucerne University of Applied 
 * Sciences and Arts (HSLU). You may not use, copy or modify this software, 
 * except in accordance with the license agreement you entered into with HSLU. 
 * For details see accompanying license terms. 
 */

package ch.hslu.cm.simulation;

import ch.randelshofer.util.*;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.io.IOException;
import javax.swing.event.*;
import java.util.*;
import org.jhotdraw.xml.DOMInput;
import org.jhotdraw.xml.DOMOutput;
/**
 * A simulated element which can have attributes.
 * <p>
 * FIXME - Move this eventually into package ch.hslu.cm.simulation.
 *
 * @author  Werner Randelshofer
 * @version 1.0 2006-03-20 Created.
 */
public abstract class AttributedElement<T extends SimulatedAttribute> extends AbstractElement {
    /**
     * T list.
     */
    protected ArrayList<T> attributes = new ArrayList<T>();
    
    /**
     * T handler class.
     */
    protected static class AttributeHandler<T extends SimulatedAttribute> implements PropertyChangeListener {
        private AttributedElement<T> owner;
        public AttributeHandler(AttributedElement<T> owner) {
            this.owner = owner;
        }
        public void propertyChange(PropertyChangeEvent evt) {
            owner.fireAttributeChanged(owner.getAttributeIndex((T) evt.getSource()));
        }
    }

    /**
     * T handler instance.
     */
    protected AttributeHandler attributeHandler = new AttributeHandler(this);
    
    
    /** Creates a new instance. */
    public AttributedElement() {
    }
    
    protected abstract T createAttribute();
    
    public void addAttribute(String name) {
        T t = createAttribute();
        t.setName(name);
        addAttribute(t);
        
    }
    public void addAttribute(T attr) {
        addAttribute(getAttributeCount(), attr);
    }
    public void addAttribute(int index, T attr) {
        attributes.add(index, attr);
        attr.addPropertyChangeListener(attributeHandler);
        fireAttributeAdded(index);
    }
    
    public boolean containsAttribute(T a) {
        return attributes.contains(a);
    }
    
    public boolean isAttribute(String name) {
        // FIXME - linear search! We should use a map to improve the performance
        
        for (Iterator i=attributes.iterator(); i.hasNext(); ) {
            T attr = (T) i.next();
            if (attr.getAttributeName().equals(name)) return true;
        }
        
        return false;
    }
    
    public T getAttribute(String name) {
        // FIXME - linear search! We should use a map to improve the performance
        int p = name.indexOf('=');
        if (p != -1) name = name.substring(0, p);
        for (Iterator i=attributes.iterator(); i.hasNext(); ) {
            T attr = (T) i.next();
            if (attr.getAttributeName().equals(name)) return attr;
        }
        
        return null;
    }
    
    public int getAttributeIndex(T attr) {
        return attributes.indexOf(attr);
    }
    
    public T getAttribute(int index) {
        return (T) attributes.get(index);
    }
    
    public T removeAttribute(int index) {
        T removed = (T) attributes.remove(index);
        removed.removePropertyChangeListener(attributeHandler);
        fireAttributeRemoved(index, removed);
        return removed;
    }
    
    public Collection<T> getAttributes() {
        return Collections.unmodifiableList(attributes);
    }
    
    public int getAttributeCount() {
        return attributes.size();
    }
    
    public void adddAttributedElementListener(AttributedElementListener l) {
        listenerList.add(AttributedElementListener.class, l);
    }
    
    public void removeAttributedElementListener(AttributedElementListener l) {
        listenerList.remove(AttributedElementListener.class, l);
    }
    
    /**
     * Notify all listeners that have registered interest for
     * notification on this event type.
     */
    protected void fireAttributeAdded(int index) {
        AttributedElementEvent event = null;
        
        // Guaranteed to return a non-null array
        Object[] listeners = listenerList.getListenerList();
        // Process the listeners last to first, notifying
        // those that are interested in this event
        for (int i = listeners.length-2; i>=0; i-=2) {
            if (listeners[i]==AttributedElementListener.class) {
                // Lazily create the event:
                if (event == null)
                    event = new AttributedElementEvent(this, index, getAttribute(index));
                ((AttributedElementListener)listeners[i+1]).attributeAdded(event);
            }
        }
    }
    /**
     * Notify all listeners that have registered interest for
     * notification on this event type.
     */
    protected void fireAttributeRemoved(int index, T attribute) {
        AttributedElementEvent event = null;
        
        // Guaranteed to return a non-null array
        Object[] listeners = listenerList.getListenerList();
        // Process the listeners last to first, notifying
        // those that are interested in this event
        for (int i = listeners.length-2; i>=0; i-=2) {
            if (listeners[i]==AttributedElementListener.class) {
                // Lazily create the event:
                if (event == null)
                    event = new AttributedElementEvent(this, index, attribute);
                ((AttributedElementListener)listeners[i+1]).attributeRemoved(event);
            }
        }
    }
    /**
     * Notify all listeners that have registered interest for
     * notification on this event type.
     */
    protected void fireAttributeChanged(int index) {
        AttributedElementEvent event = null;
        
        // Guaranteed to return a non-null array
        Object[] listeners = listenerList.getListenerList();
        // Process the listeners last to first, notifying
        // those that are interested in this event
        for (int i = listeners.length-2; i>=0; i-=2) {
            if (listeners[i]==AttributedElementListener.class) {
                // Lazily create the event:
                if (event == null)
                    event = new AttributedElementEvent(this, index, getAttribute(index));
                ((AttributedElementListener)listeners[i+1]).attributeChanged(event);
            }
        }
    }
    
    public AttributedElement clone() {
        AttributedElement that = (AttributedElement) super.clone();
        
        // We have to do a deep clone of the attributes, because
        // the identity of an attribute is used to determine to which
        // entity it belongs.
        that.attributes = new ArrayList<T>();
        that.attributeHandler = new AttributeHandler(that);
        for (T thisAttr : attributes) {
            that.addAttribute((T) thisAttr.clone());
        }
        return that;
    }
    
    public void write(DOMOutput out) throws IOException {
        if (attributes.size() > 0) {
            out.openElement("attributes");
            for (Iterator i=attributes.iterator(); i.hasNext(); ) {
                out.writeObject(i.next());
            }
            out.closeElement();
        }
    }
    public void read(DOMInput in) throws IOException {
        if (in.getElementCount("attributes") > 0) {
            in.openElement("attributes");
            for (int i=0; i < in.getElementCount(); i++) {
                addAttribute((T) in.readObject(i));
            }
            in.closeElement();
        }
    }
}
